%-------------------------------------------------------------------%
%            Spectral unmixing under linear constraints             %
%                                                                   %
%-------------------------------------------------------------------%

clc
close all
clear all
addpath data
addpath include

%% Generate and save synthetic data

load library.mat
lst = names;

m = 100; % Image size (nxm)
n = m;
N = m*n; % Total number of pixels 
P = 5; % Spectra number between 1 and 21

seed = 1;
randn('state',seed);
rand('state',seed);

disp(' ')
disp('---Generate synthetic data---')
disp([' 1. Image size: ',num2str(m),' x ',num2str(n) ' pixels'])
disp([' 2. Create abundance maps'])
F = 30; % number of patterns in the image
C = gen_abundance(m,n,P,F);

disp([' 3. Select ',num2str(P),' spectra from the library'])
ord   = randperm(size(spec,2));
S     = spec(:,ord(1:P));
names = lst(ord(1:P),:);
for p = 1:P
    namestr(p,:) = sprintf('%s',char(names(p,:)));
end
figure('Name','Reflectance spectra');
plot(wav,S)
legend(namestr)
xlabel('\mu m');
ylabel('reflectance');
axis tight

disp(' 4. Add Gaussian noise after linear mixing')
Y     = S*reshape(C,N,P)';
L     = size(Y,1);

SNR = 20;
Y   = add_noise(SNR,Y,'g');
img = reshape(Y,L,m,m);

disp(' 5. Save synthetic data ')
save(['data' filesep 'simu' num2str(m) 'x' num2str(n) '_' num2str(P) '.mat'],'img','wav','C','S','names','-mat'); 

%% Perform spectral unmixing
disp(' ')
disp('---Perform the spectral unmixing by IPLS---')

% 'constraint'    constraint type
%   'sto'       positivity and sum to one (default)
%   'slo'       positivity and sum less than one
%   'pos'       positivity

%  'hesscomp'      hessian computing strategy
%    'pixelwise' uses less memory (default)
%    'global'    faster but requires maximal memory
%    'blockwise' between pixelwise and global

%%
%
disp('---')
disp('A. Non-negativity constraint')
disp('---')
constraint = 'pos';

% pixelwise processing
fprintf(' 1. Pixelwise processing   ')
hesscomp = 'pixelwise';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])
%disp(' ')

% fullwise processing
fprintf(' 2. Image-based processing ')
hesscomp = 'global';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])

%disp(' ')

% blockwise
fprintf(' 3. Blockwise processing   ')
hesscomp = 'blockwise';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])
disp(' ')

%%
disp('---')
disp('B. Non-negativity and sum less than one constraints')
disp('---')
constraint = 'slo';
%
% pixelwise processing
fprintf(' 1. Pixelwise processing   ')
hesscomp = 'pixelwise';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])
%disp(' ')

% fullwise processing
fprintf(' 2. Image-based processing ')
hesscomp = 'global';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])

%disp(' ')

% blockwise
fprintf(' 3. Blockwise processing   ')
hesscomp = 'blockwise';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])
disp(' ')
%%
disp('---')
disp('C. Non-negativity and sum equal to one constraints')
disp('---')
constraint = 'sto';
%
% pixelwise processing
fprintf(' 1. Pixelwise processing   ')
hesscomp = 'pixelwise';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])
%disp(' ')

% fullwise processing
fprintf(' 2. Image-based processing ')
hesscomp = 'global';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])

%disp(' ')

% blockwise
fprintf(' 3. Blockwise processing   ')
hesscomp = 'blockwise';
tic;
[Cest,Cstd,Mem] = ipls(Y, S, 'hesscomp', hesscomp, 'constraint', constraint);
Tps = toc;
Cest = reshape(Cest',m,n,P);
NMSE = 100/P* sum(sum(sum((C(:,:,:)-Cest(:,:,:)).^2,1),2)./sum(sum(C(:,:,:).^2,1),2));
disp(['    NMSE (%) = ',num2str(NMSE), ' |  Time (s) = ',num2str(Tps) '| Memory (MB) ', num2str(Mem) ])
disp(' ')

%% Display result

p = ceil(P*rand); %component number between 1 and P
figure('Name',namestr(p,:))
subplot(121)
imagesc(C(:,:,p),[ 0 1]);
title('Original map')
 axis image off
subplot(122)
imagesc(Cest(:,:,p),[0 1])
title(['Restored map with NMSE (%) = ',num2str(NMSE)])
axis image off
 

