function [C, Cstd, mem] = ipls(Y, S, varargin)

%
% Interior Point Least Squares Optimization
%
% Inputs
%   Y           Hyperspectral image [ nb_bands x nb_pixels ]
%   S           Spectral library [ nb_bands x nb_endmembers ]
%   options     cov             observation covariance matrix [ nb_bands x nb_bands ]
%               'constraint'    constraint type
%                   'sto'       positivity and sum to one (default)
%                   'slo'       positivity and sum less than one
%                   'pos'       positivity
%               'hesscomp'      hessian computing strategy
%                   'pixelwise' uses less memory (default)
%                   'global'    faster
%                   'blockwise' between pixelwise and global
%               'blocksize',LK  set the blocksize to LK in case of a 'blockwise' hessian computing (default: 1024)
%
% Outputs
%   C           Abundances [ nb_endmembers x nb_pixels ]
%   Cstd        Abundances standard deviation [ nb_endmembers x nb_pixels ]
%   mem         Memory usage (MB)
%
% Examples
%   C = ipls(Observation, Lib, Covariance, 'hesscomp', 'global', 'constraint', 'slo')
%   [C, Cstd] = ipls(Observation, Lib, 'constraint', 'pos')
%



constraint = 1;   % 1 for 'sto' ; 2 for 'slo' ; 3 for 'pos'
hesscomp   = 1;   % 1 for 'pixelwise' ; 2 for 'global' ; 3 for 'blockwise'
inputCov   = 0;   % boolean
LK = 1024; %

L = size(Y,1);
% Optional input management

i = 1;
while i<=size(varargin,2)
    if all(size(varargin{i})==[L,L])
        inputCov = 1;
        Cov = varargin{i};
    elseif strcmp(varargin{i},'constraint')
        i = i + 1;
        switch varargin{i}
            case 'sto'
                constraint = 1;
            case 'slo'
                constraint = 2;
            case 'pos'
                constraint = 3;
            otherwise
                disp('Error in IPLS : Invalid constraint')
                return
        end
    elseif strcmp(varargin{i},'hesscomp')
        i = i + 1;
        switch varargin{i}
            case 'pixelwise'
                hesscomp = 1;
            case 'global'
                hesscomp = 2;
            case 'blockwise'
                hesscomp = 3;
            otherwise
                disp('Error in IPLS : Invalid hesscomp')
                return
        end
    elseif strcmp(varargin{i},'blocksize')
        i = i + 1;
        LK=varargin{i};
    else
        disp(['Error in IPLS : Argument ' num2str(i+2) ' is invalid'])
        return
    end
    i = i + 1;
end
% Inputs rescaling with observation covariance

if inputCov
    U = chol(inv(Cov));
    Y = U*Y;
    S = U*S;
end

% Algorithm design parameters.

maxiter   = 1000;
tolerance = 1e-8;
mumin     = 1e-9;   % Minimum barrier parameter.
alphamax  = 0.995;  % Maximum step size.
beta      = 0.75;   % Granularity of backtracking search.
tau       = 0.01;   % Wolfe parameter of backtracking search
etadf     = 100;
etac      = 1.9;
rho       = 2;

% Initialization

P = size(S,2);  % number of endmembers
N = size(Y,2);  % number of pixels
Rs = S'*S;

if constraint == 1
    Cot  = [eye(P-1) zeros(P-1,1)] - [zeros(P-1,1) eye(P-1)];
    Co   = Cot';
    co   = 1/P;
    Ynew = Y-S*ones(P,N)*co;
    Q    = 2*Cot*Rs*Co;
    q    = -2*Co'*S'*Ynew;
    X   = zeros(P-1,N);     % primal variable
    Z   = ones(P,N);        % dual variable
    PX  = zeros(P-1,N);     % primal direction
    n   = N*(P-1);
    m   = N*P;
elseif constraint == 2
    Cot  = [speye(P) -ones(P,1)];
    Co   = Cot';
    co   = [zeros(P,1) ; 1]*ones(1,N);
    Q    = 2*Rs;
    q    = -2*S'*Y;
    X   = ones(P,N)/(P+1);     % primal variable
    Z   = ones(P+1,N);        % dual variable
    PX  = zeros(P,N);     % primal direction
    n   = N*P;
    m   = N*(P+1);
elseif constraint == 3
    Cot  = speye(P);
    Co   = Cot';
    co   = zeros(P,N);
    Q    = 2*Rs;
    q    = -2*S'*Y;
    X   = ones(P,N)/(P+1);     % primal variable
    Z   = ones(P,N);        % dual variable
    PX  = zeros(P,N);     % primal direction
    n   = N*P;
    m   = N*P;
end

if hesscomp == 2
    Cobig = kron(speye(N),Co);
    Qbig = kron(speye(N),Q);
elseif hesscomp == 3
    if N~=1,
        KK = floor(N/LK);
    else
        KK = 1;
    end
    lK = N-KK*LK;
    Lm = m/N*LK;
    Ln = n/N*LK;
    %
    Qbig  = kron(speye(LK),Q);
    Cobig = kron(speye(LK),Co);
    %
    Qsmall  = kron(speye(lK),Q);
    Cosmall = kron(speye(lK),Co);
end

CoX = Co*X;
C   = CoX + co;         % constraints
nv = n + m;

QX    = Q*X;
f     = q(:)'*X(:) + 1/2*X(:)'*QX(:); % objective function
invC  = 1./C;
RC    = C.*Z; % Complementarity.
CtZ   = sum(RC(:));

dualitygap = CtZ;
mu = max(mumin,(dualitygap/m)/rho);


for iter = 1:maxiter
    % barrier parameter update
    G  = q + QX; % Gradient
    RX = G - Cot*Z; % Dual residual.
    r0 = [RX(:) ; -RC(:)];

    % Set some parameters that affect convergence of the primal-dual
    % interior-point method.
    rx = RX(:);
    dualitygap = sum(RC(:));

    if(iter>1)
        if(max(abs(rx))<etadf*mu && dualitygap/m<etac*mu)
            mu = max(mumin,(dualitygap/m)/rho);
        end
    end

    % CONVERGENCE CHECK.
    if (norm(r0)/nv < tolerance),  break, end
    if mu == mumin, break,  end

    % SOLUTION TO PERTURBED KKT SYSTEM.
    Ssmall = Z./C;
    Gz     = - mu*Cot*invC;
    GB     = G + Gz;
    if hesscomp ==1 %pixelwise
        for n=1:N
            Hk = Q + Cot*diag(Ssmall(:,n))*Co;
            PX(:,n) = Hk\(-GB(:,n));
        end
    elseif hesscomp == 2 % full image
        Sbig = diag(sparse(Ssmall(:)));
        Hess = Qbig + Cobig'*Sbig*Cobig;
        PX(:) = Hess \ (-GB(:));
        %PX = reshape(PX, size(Q,1),N); %primal direction
    elseif hesscomp == 3 % blockwise
        S    = Ssmall(:);
        GB   = GB(:);
        %
        for k=1:KK
            Hess = Qbig + Cobig'*diag(sparse(S((k-1)*Lm+1:k*Lm)))*Cobig;
            PX((k-1)*Ln+1:k*Ln) = Hess\(-GB((k-1)*Ln+1:k*Ln));
        end
        %
        if lK~=0
            Hess = Qsmall + Cosmall'*diag(sparse(S(KK*(m/N*LK)+1:end)))*Cosmall;
            PX(KK*(LK*n/N)+1:end) = Hess\(-GB(KK*LK*(n/N)+1:end));
        end
    end

    dX = Co*PX;
    PZ = -(Z + Ssmall.*dX  - mu.*invC); % dual direction

    % BACKTRACKING LINE SEARCH.
    delta = [dX(:) ; PZ(:)];
    theta = [C(:) ; Z(:)];
    alpha = min(-theta(delta<0)./delta(delta<0)) ; %Position of barrier
    if(isempty(alpha))
        alpha = alphamax;
    else
        alpha = alphamax*min(1,alpha);
    end

    % Compute the response of the merit function and the directional
    % gradient at the current point and search direction.
    psi  = f + CtZ - mu*sum(log(C(:).*RC(:)));
    dpsi = PX(:)'*(2*GB(:) - RX(:)) - PZ(:)'*(mu./Z(:) - C(:));

    ls   = 0;
    while true
        % Compute the candidate point, the constraints, and the response of
        % the objective function and merit function at the candidate point.
        ls      = ls + 1;
        Xnew    = X + alpha * PX;
        Znew    = Z + alpha * PZ;
        QX      = Q*Xnew;
        f       = q(:)'*Xnew(:) + 1/2*Xnew(:)'*QX(:);
        CoXnew  = CoX + alpha*dX;
        C       = CoXnew + co;
        RC      = C.*Znew;
        CtZ     = sum(RC(:));
        psinew  = f + CtZ - mu*sum(log(C(:).*RC(:)));
        %
        % Stop backtracking search if we've found a candidate point that
        % sufficiently decreases the merit function and satisfies all the
        % constraints.
        if (all(C(:)>0) && psinew < psi + tau*alpha*dpsi) || ls>=20
            X       = Xnew;
            Z       = Znew;
            invC    = 1./C;
            CoX     = CoXnew;
            break
        end
        % The candidate point does not meet sufficient convergence criteria, so decrease the step
        % size for 0 < beta < 1.
        alpha = alpha * beta;
    end

end

if constraint == 2,
    C(P+1,:) = []; 
end

if nargout >= 2
    Cstd = zeros(P,N);
    for n=1:N
        Hc = 2*Rs + diag(Ssmall(1:P,n));
        Cstd(:,n) = sqrt(diag(inv(Hc)));
    end
else 
    Cstd=[];
end

if nargout >= 3
    vv=whos;
    mem=0;
    for k=1:length(vv)
        mem=mem+vv(k).bytes;
    end
    mem = mem*1e-6;
end

end

