%-------------------------------------------------------------%
% Minimization of logarithmic barrier function for QP problem %                                                   %
%-------------------------------------------------------------%

%The following QP problem is considered:
%min_x 1/2 x^T A x + a^T x + a0 s.t.  C x + c >= 0 (*)
%
%According to the interior-point theory, the barrier function
%associated to (*) is
%
%F(x) = 1/2 x^T A x + a^T x + a0 - mu sum(log(C x + c))
%
%with mu>0 the barrier parameter.
%
%This program allows to minimize F using a Newton algorithm
%associated to a Majorize-Minimize linesearch strategy.

clear all
close all
warning off

%Set QP problem parameters (see [More et al,89])
n = 100;
ncond = 12;
ndeg = 5;
nact = 50;
nact0 = 0;
seed = 0;

%Generate the QP problem
[A,a,a0,C,c,x0,xstar] = QuadPbMore(n,ncond,ndeg,nact,nact0,seed);
m = size(C,1);
%Set the barrier parameter
mu = 1e-4;

%Objective function
F = @(x) 1/2.*x'*A*x + a'*x +a0 - mu*sum(log(C*x + c));
%Gradient
dF = @(x) A*x+a-mu*C'*(1./(C*x+c));
%Hessian
d2F = @(x) A + mu*C'*diag(1./(C*x+c).^2)*C;

%Number of MM linesearch subiterations
J = [1 2 5 10];
Nj = length(J);
%Number of Newton algorithm iterations
K = 50;

Crit = zeros(K,Nj);
Grad = zeros(K,Nj);
Time = zeros(K,Nj);

for jj=1:Nj
    disp('-------')
    disp(['J = ' num2str(J(jj))])
    disp('-------')

    %Initialization
    x = x0;
    tic
    for k = 1:K
        %Computation of Newton direction
        g = dF(x);
        H = d2F(x);
        d = - H\g;
        %Computation of the stepsize
        mp = @(alpha) d'*A*d;
        df = @(alpha)d'* dF(x + alpha*d);
        alpha = MMBarrierLineSearch(mp,mu.*ones(m,1),C*x+c,C*d,df,J(jj));
        %Update
        x = x+alpha*d;
        
        %Display
        Time(k,jj) = toc;
        Crit(k,jj) = F(x);
        Grad(k,jj) = norm(dF(x));
        disp(['k = ',num2str(k),': F(x_k) = ',num2str(Crit(k,jj))]);
    end
    Jlist{jj}=['J = ' num2str(J(jj))];
end

figure

subplot(221)
plot(1:K,Crit,'-x','linewidth',2);
xlabel('Iteration');
title('Objective function value');
legend(Jlist)
%
subplot(222)
semilogy(1:K,Grad,'-x','linewidth',2);
xlabel('Iteration');
title('Objective function gradient norm');
legend(Jlist)

subplot(223)
plot(Time,Crit,'-x','linewidth',2);
xlabel('Time');
title('Objective function value');
legend(Jlist)
%
subplot(224)
semilogy(Time,Grad,'-x','linewidth',2);
xlabel('Time');
title('Objective function gradient norm');
legend(Jlist)