function alpha = MMBarrierLineSearch(mp,kappa,theta,delta,df,J)

% ========================================================================
% MM barrier line search algorithm, Version 1.0
%
% Kindly report any suggestions or corrections to
% emilie.chouzenoux@univ-mlv.fr
%
%----------------------------------------------------------------------
% This code performs J iterations of the Majorize-Minimize barrier linesearch 
% proposed in:
%
% E. Chouzenoux, S. Moussaoui and J. Idier. "Majorize-Minimize Linesearch for Inversion 
% Methods Involving Barrier Function Optimization." Inverse Problems, Vol. 28, No. 6, 2012.
%
% Let an iterative descent algorithm x <- x + alpha d for minimizing a criterion F. 
% The linesearch will determine an approximate minimizer of f(alpha) = F(x + alpha d) 
% that will ensures the convergence of the descent algorithm.
%
% Here, we assume that f takes the following form:
%
% f(alpha) = p(alpha) - sum_i kappa(i) log(theta(i) + alpha * delta(i))
%
% Moreover, we assume that there exists mp(alpha)>=0 such that, for all real alpha,alpha',
%
% p(alpha') <= p(alpha) + (alpha'-alpha)*dp(alpha) + 1/2 * mp(alpha) * (alpha'-alpha)^2 
%----------------------------------------------------------------------
% Input:    mp : handle function s.t. mp(alpha) is the curvature of a quadratic
%           majorant of p at alpha
%           kappa,theta,delta : logarithmic barrier function parameters
%           df : handle function s.t. df(alpha) equals to the derivative of f at alpha.
%           J  : number of MM iterates
%
% Output:    alpha: result of the linesearch
%----------------------------------------------------------------------

%Initialization
alpha = 0;
mp_alpha = mp(alpha);
df_alpha = df(alpha);

for j = 1:J
    [mb,gam,alpha_minus,alpha_plus]=CoefMajorLog(kappa,theta,delta,alpha) ;
    if df_alpha<=0 %The minimizer is greater than alpha
         M = mp_alpha + mb(2) ; Gam= gam(2) ; %Right part of the tangent majorant
         if(isinf(alpha_plus)) %No barrier in the right side
                alpha  = alpha- df_alpha/M ;
         else
                q1 = -M ;
                q2 = Gam-df_alpha+M* (alpha_plus-alpha)  ;
                q3 =  (alpha_plus-alpha) *df_alpha ;

                alpha  = alpha -(2*q3)/(q2+sqrt(q2^2-4*q1*q3)) ;
         end
    else %The minimizer is lower than alpha
        M = mp_alpha  + mb(1) ; Gam= gam(1) ; %Left part of the tangent majorant
        if(isinf(alpha_minus)) %No barrier in the left side
            alpha = alpha - df_alpha/M ;
        else
           q1 = -M;
           q2 = Gam-df_alpha+M*(alpha_minus-alpha);
           q3 = (alpha_minus-alpha)*df_alpha;
           alpha = alpha-(2*q3)/(q2-sqrt(q2^2-4*q1*q3)) ;
        end

    end
    if(j<J)
        mp_alpha = mp(alpha);
        df_alpha = df(alpha);
    end

end