



function [x,Crit,NGrad] = MajorizeMinimizeMemoryGradient(y,H,H_adj,eta,tau,lambda,delta,xmin,xmax,phi,x,Nx,Ny,NbIt)

% ========================================================================
% MM-MG Algorithm, Version 1.0
%
% Kindly report any suggestions or corrections to
% emilie.chouzenoux@univ-mlv.fr
%
%----------------------------------------------------------------------
%
%Input:  y: the degraded data
%        H: the linear degradation operator (function handle)
%        H_adj: the adjoint of the linear degradation operator
%        (function handle)
%        tau, lambda, delta: the regularization parameters
%        xmin, xmax: the bounds 
%        phi: the penalty function flag as indicated below
%           (1) phi(u) = (1-exp(-u.^2./(2*delta^2))); 
%           (2) phi(u) = (u.^2)./(2*delta^2 + u.^2); 
%           (3) phi(u) = log(1 + (u.^2)./(delta^2)); 
%           (4) phi(u) = sqrt(1 + u^2/delta^2)-1; 
%           (5) phi(u) = 1/2 u^2; 
%       NbIt: the max number of iterations.  

%Output: x: the restored image
%        Crit: the values of the criterion along minimization process
%        NGrad: the value of gradient norm along minimization process
 

%Minimization of
%
%F(x) = 1/2 || H(x)- y ||^2 + lambda sum(phi(V(x)))
%       + tau ||x||^2 + 1/2 eta ||Proj_[xmin;xmax]-x||^2
%
%========================================================================

stop = min(Nx,Ny)*1e-4; 
disp('****************************************');
disp('Majorize-Minimize Memory Gradient Algorithm');

switch phi
    case 1
            disp('phi(u) =  (1-exp(-u^2/(2*delta^2)))');
    case 2
            disp('phi(u) = (u^2)/(2*delta^2 + u^2)');
    case 3
            disp('phi(u) = log(1 + (u^2)/(delta^2))');
    case 4
            disp('phi(u) =  sqrt(1 + u^2/delta^2)-1');
    case 5        
            disp('phi(u) = 1/2 u^2'); 
end

disp(['lambda = ',num2str(lambda),', delta = ',num2str(delta),', eta = ',num2str(eta),' and tau = ',num2str(tau)]);
disp(['xmin = ',num2str(xmin),' and xmax = ',num2str(xmax)]);

NGrad = zeros(NbIt,1);

[Crit(1),Grad,Vx] = critere(x,y,H,H_adj,eta,tau,lambda,delta,phi,xmin,xmax,Nx,Ny);
disp(['Initial criterion value = ',num2str(Crit(1))])
Time(1) = 0;
NGrad(1) = norm(Grad);

 

for k = 1:NbIt
   
    tic
    
    %Stopping test
    if(NGrad(k)<stop)
        break
    end
    
     
    Vg = Voperator(Grad,0,Nx,Ny);
    Hg = H(Grad);
    
    if(k==1)
        
        Dir   = -Grad;         
        B = majorantem(Vx,-Vg,-Hg,Dir,eta,tau,lambda,delta,phi);
        s = (Grad'*Grad)/B;
        dx = Dir*s;
        Vdx = -s*Vg;
        Hdx = -s*Hg;
        
    else
        
        Dir = [-Grad dx];
        VD = [-Vg Vdx];
        HD = [-Hg Hdx];
        B = majorantem(Vx,VD,HD,Dir,eta,tau,lambda,delta,phi);
        s = -pinv(B)*(Dir'*Grad);
        dx = Dir*s;
        Vdx = VD*s;
        Hdx = HD*s;      
        
     end
     
    %update
    x = x + dx;
   
    [Crit(k+1),Grad,Vx] = critere(x,y,H,H_adj,eta,tau,lambda,delta,phi,xmin,xmax,Nx,Ny);
    
    Time(k+1) = toc;
    NGrad(k+1) = norm(Grad);

    
end
disp(['Iteration number = ',num2str(length(Crit))]);
disp(['Computation time (cpu) =', num2str(sum(Time))]);
disp(['Final criterion value = ',num2str(Crit(end))])
disp('****************************************');
end





function [F,dF,Vx] = critere(x,y,H,H_adj,eta,tau,lambda,delta,phi,xmin,xmax,Nx,Ny)

Vx = Voperator(x,0,Nx,Ny);
Hx = H(x);

UN = ones(length(x),1);
UNmin = UN;
UNmin(x>xmin)=0;
UNmax = UN;
UNmax(x<xmax)=0;

switch phi
    case 1
        phiVx =  (1-exp(-(Vx.^2)./(2*delta^2)));
        dphiVx =  (Vx./(delta^2)).*exp(-(Vx.^2)./(2*delta^2));
    case 2
        phiVx = (Vx.^2)./(2*delta^2 + Vx.^2);
        dphiVx = (4.*Vx.*(delta^2))./(2*delta^2+(Vx.^2)).^2;
    case 3
        phiVx = log(1 + (Vx.^2)./(delta^2));
        dphiVx = (2.*Vx)./(delta^2+(Vx.^2));
    case 4
        phiVx = (1 + (Vx.^2)./(delta^2)).^(1/2)-1;
        dphiVx = (Vx./(delta^2)).*(1+(Vx.^2)/delta^2).^(-1/2); 
    case 5
        phiVx = 1/2 .*Vx.^2;
        dphiVx = Vx;
end

F = 1/2 * (Hx-y)'*(Hx-y) + tau * (x'*x) + lambda * sum(phiVx) ;
dJ = H_adj(Hx-y);
dF = dJ + 2*tau*x+ lambda.*Voperator(dphiVx,1,Nx,Ny);

if(~isinf(xmin))
    F = F + eta*sum(1/2*(x(x<xmin)-xmin).^2);
    dF = dF + eta.*(x-xmin).*UNmin;
end
if(~isinf(xmax))
    F = F + eta*sum(1/2*(x(x>xmax)-xmax).^2);
    dF = dF + eta.*(x-xmax).*UNmax;
end
 
end

function B = majorantem(Vx,VD,HD,D,eta,tau,lambda,delta,phi)
[n,m] = size(D);




switch phi
    case 1
        wVx = (1/(delta^2)).*exp(-(Vx.^2)./(2*delta^2)); 
    case 2
        wVx =  (4*delta^2)./(2*delta^2 + Vx.^2).^2;
    case 3
        wVx = 2./(delta^2 + Vx.^2);
    case 4
        wVx = (1/(delta^2)).*(1+(Vx.^2)/delta^2).^(-1/2);
    case 5       
        wVx = ones(length(Vx),1);
end

DthtHD = (HD)'*(HD);
DtD = D'*D;
DtVtWVD = VD'*(wVx(:,ones(m,1)).*VD);
B = DthtHD + 2*tau*DtD + lambda.*DtVtWVD + 2*eta.*DtD;
 

end


function Vx = Voperator(x,adj,Nx,Ny)

hv = [0 -1 0;0 1 0;0 0 0];%vertical gradient
hh = [0 0 0 ;-1 1 0; 0 0 0];%horizontal gradient

if(adj==0)%Vx = V*x
    Vv = reshape(conv2(reshape(x,Nx,Ny),hv,'same'),Nx*Ny,1);
    Vh = reshape(conv2(reshape(x,Nx,Ny),hh,'same'),Nx*Ny,1);
    Vx = [Vv;Vh];              
else%Vx = V'*x
    xv = x(1:Nx*Ny);
    xh = x(Nx*Ny+1:2*Nx*Ny);
    Vtv = reshape(conv2(reshape(xv,Nx,Ny),fliplr(flipud(hv)),'same'),Nx*Ny,1);
    Vth = reshape(conv2(reshape(xh,Nx,Ny),fliplr(flipud(hh)),'same'),Nx*Ny,1);
    Vx = Vtv + Vth;
end

end
